<?php

if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}

function startAddProductFlow(int $chat_id, int $user_id, string $platform): void {
    $session_data = ['state' => 'awaiting_product_type', 'add_product_data' => []];
    updateUserField($user_id, 'session', json_encode($session_data));
    $buttons = [[['text' => 'ساده'], ['text' => 'متغیر (ویژگی‌دار)']]];
    sendMessage(
        $chat_id, 
        "نوع محصولی که می‌خواهید اضافه کنید را انتخاب کنید:", 
        ['keyboard' => $buttons, 'resize_keyboard' => true, 'one_time_keyboard' => true], 
        $platform
    );
}

function handleProductType(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $message_text = trim($message['text'] ?? '');
    $type = null;
    if ($message_text === 'ساده') {
        $type = 'simple';
    } elseif ($message_text === 'متغیر (ویژگی‌دار)') {
        $type = 'variable';
    }

    if ($type) {
        $session_data['add_product_data']['type'] = $type;
        $session_data['state'] = 'awaiting_category_selection';
        updateUserField($user_id, 'session', json_encode($session_data));
        askForCategory($chat_id, $user_id, $platform, $session_data, null);
    } else {
        sendMessage($chat_id, "لطفا نوع محصول را با استفاده از دکمه‌های نمایش داده شده انتخاب کنید.", null, $platform);
    }
}

function askForCategory(int $chat_id, int $user_id, string $platform, array $session_data, ?int $message_id, ?int $parent_id = null): void {
    $can_edit = ($message_id !== null);
    $categories = getActiveCategories($parent_id);
    $buttons = [];
    $path_text = "ریشه";

    if ($parent_id !== null) {
        $path_array = getCategoryPath($parent_id);
        $path_names = array_map(function($p) { return htmlspecialchars($p['name'] ?? '?'); }, $path_array);
        $path_text = implode(" > ", $path_names);
    }

    $message_text_body = "شما در مسیر *{$path_text}* هستید.\n\nلطفا دسته‌بندی محصول را انتخاب کنید:";

    if (!empty($categories)) {
         foreach ($categories as $cat) {
            $buttons[] = [['text' => '📁 ' . $cat['name'], 'callback_data' => buildCallbackData('setCat', ['id' => $cat['id']])]];
        }
    }
    
    $buttons[] = [['text' => '✅ انتخاب این دسته', 'callback_data' => buildCallbackData('confirmCat', ['id' => $parent_id])]];
    
    if ($parent_id !== null) {
        $parent_cat_info = getCategoryById($parent_id);
        $back_target_id = $parent_cat_info['parent_id'] ?? null;
        $buttons[] = [['text' => '⬆️ بازگشت به سطح قبل', 'callback_data' => buildCallbackData('setCat', ['id' => $back_target_id])]];
    }

    $reply_markup = ['inline_keyboard' => $buttons];
    
    if ($can_edit) {
        editMessage($chat_id, $message_id, $message_text_body, $reply_markup);
    } else {
        sendMessage($chat_id, $message_text_body, $reply_markup, $platform);
    }
}

function handleCategoryNavigation(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $category_id = (int)($params['id'] ?? 0);
    $category_id = ($category_id === 0) ? null : $category_id;
    
    $session_data['state'] = 'awaiting_category_selection';
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForCategory($chat_id, $user_id, $platform, $session_data, $message_id, $category_id);
}

function confirmCategoryAndProceed(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $category_id = (int)($params['id'] ?? 0);
    $category_id = ($category_id === 0) ? null : $category_id;

    $session_data['add_product_data']['category_id'] = $category_id;
    $session_data['state'] = 'adding_product_photo';
    updateUserField($user_id, 'session', json_encode($session_data));

    if ($message_id) {
        apiRequest('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id], $platform);
    }

    $category_name = "بدون دسته";
    if ($category_id !== null) {
        $category = getCategoryById($category_id);
        if ($category) {
            $category_name = $category['name'];
        }
    }

    sendMessage(
        $chat_id, 
        "دسته‌بندی '{$category_name}' انتخاب شد. اکنون عکس اصلی محصول را ارسال کنید:", 
        ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
        $platform
    );
}

function handleProductPhoto(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    if (isset($message['photo'])) {
        $photo_array = $message['photo'];
        $source_file_id = end($photo_array)['file_id'];
        sendMessage($chat_id, "⏳ در حال پردازش و ذخیره‌سازی تصویر...", null, $platform);
        
        $file_content = downloadFileContent($source_file_id, $platform);

        if ($file_content) {
            try {
                if (!defined('PRODUCT_IMAGE_STORAGE_PATH')) {
                    throw new Exception("PRODUCT_IMAGE_STORAGE_PATH is not defined.");
                }
                $storage_path = PRODUCT_IMAGE_STORAGE_PATH;
                if (!is_dir($storage_path) || !is_writable($storage_path)) {
                    if (!@mkdir($storage_path, 0775, true)) {
                        throw new Exception("File storage path is not writable or cannot be created: {$storage_path}");
                    }
                }
                $finfo = new finfo(FILEINFO_MIME_TYPE);
                $mime_type = $finfo->buffer($file_content) ?: 'image/jpeg';
                $extension = 'jpg';
                if (strpos($mime_type, 'png') !== false) $extension = 'png';
                elseif (strpos($mime_type, 'gif') !== false) $extension = 'gif';
                elseif (strpos($mime_type, 'webp') !== false) $extension = 'webp';
                
                $new_filename = 'prod_' . uniqid('', true) . '.' . $extension;
                $local_save_path = rtrim($storage_path, '/') . '/' . $new_filename;

                if (file_put_contents($local_save_path, $file_content) === false) {
                    unset($file_content);
                    throw new Exception("Failed to write temporary file: {$local_save_path}");
                }
                unset($file_content);
                
                $session_data['add_product_data']['image_path'] = $new_filename;
                $session_data['state'] = 'awaiting_product_name';
                updateUserField($user_id, 'session', json_encode($session_data));
                sendMessage(
                    $chat_id, 
                    "✅ عکس با موفقیت ذخیره شد.\n\nاکنون نام محصول را وارد کنید:", 
                    ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
                    $platform
                );
            } catch (Exception $e) {
                if (defined('ERROR_LOG_PATH')) { file_put_contents(ERROR_LOG_PATH, date('Y-m-d H:i:s') . " - handleProductPhoto Save Error: " . $e->getMessage() . "\n", FILE_APPEND); }
                sendMessage($chat_id, "❌ خطا در ذخیره‌سازی عکس. لطفا دوباره تلاش کنید یا با پشتیبانی تماس بگیرید. جزئیات خطا در لاگ ثبت شد.", null, $platform);
            }
        } else {
             sendMessage($chat_id, "❌ خطا در دانلود عکس. لطفا دوباره تلاش کنید یا با پشتیبانی تماس بگیرید. جزئیات خطا در لاگ ثبت شد.", null, $platform);
        }
    } else {
        sendMessage($chat_id, "لطفا فقط عکس محصول را ارسال کنید یا انصراف دهید.", null, $platform);
    }
}

function handleProductName(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $name = trim($message['text'] ?? '');
    if (!empty($name)) {
        $session_data['add_product_data']['name'] = strip_tags($name);
        $session_data['state'] = 'awaiting_product_description';
        updateUserField($user_id, 'session', json_encode($session_data));
        sendMessage(
            $chat_id, 
            "نام ثبت شد. اکنون توضیحات محصول را وارد کنید:", 
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
            $platform
        );
    } else {
        sendMessage($chat_id, "نام محصول نمی‌تواند خالی باشد.", null, $platform);
    }
}

function handleProductDescription(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $description = trim($message['text'] ?? '');
    if (!empty($description)) {
        $session_data['add_product_data']['description'] = strip_tags($description);
        updateUserField($user_id, 'session', json_encode($session_data));
        askForProductMode($chat_id, $platform);
    } else {
        sendMessage($chat_id, "توضیحات نمی‌تواند خالی باشد.", null, $platform);
    }
}

function askForProductMode(int $chat_id, string $platform): void {
    $text = "توضیحات ثبت شد.\n\nلطفا نحوه فروش و شمارش این محصول را انتخاب کنید:";
    
    $buttons = [
        [['text' => '⚖️ فله (وزنی)', 'callback_data' => buildCallbackData('setProductMode', ['mode' => 'bulk'])]],
        [['text' => '📦 بسته‌بندی (وزن ثابت)', 'callback_data' => buildCallbackData('setProductMode', ['mode' => 'package'])]],
        [['text' => '🔢 تعدادی (بدون وزن)', 'callback_data' => buildCallbackData('setProductMode', ['mode' => 'count'])]]
    ];
    
    sendMessage($chat_id, $text, ['inline_keyboard' => $buttons], $platform);
}

function handleProductModeSelection(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $mode = $params['mode'] ?? 'count';
    $session_data['add_product_data']['mode'] = $mode;
    
    if ($message_id) {
        apiRequest('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id], $platform);
    }

    $product_type = $session_data['add_product_data']['type'] ?? 'simple';

    // تنظیمات اولیه بر اساس مد انتخاب شده
    if ($mode === 'bulk') {
        // فله: وزن ثابت نال است (چون مشتری تعیین می‌کند)
        $session_data['add_product_data']['weight'] = null;
        $session_data['add_product_data']['weight_unit'] = null;
        // نکته: واحد را در ادامه از ادمین می‌پرسیم
    } elseif ($mode === 'count') {
        // تعدادی: وزن نال
        $session_data['add_product_data']['weight'] = null;
        $session_data['add_product_data']['weight_unit'] = null;
    }
    // بسته: همه چیز پرسیده می‌شود

    if ($product_type === 'simple') {
        // برای محصول ساده، قدم بعدی قیمت است
        $session_data['state'] = 'awaiting_product_price';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        sendMessage(
            $chat_id, 
            "حالت انتخاب شد. اکنون قیمت محصول را به تومان وارد کنید (فقط عدد انگلیسی):\n(مثال: `150000`)", 
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
            $platform
        );
    } else {
        // برای محصول متغیر
        // قیمت و موجودی و وزن در ترکیب‌ها پرسیده می‌شود.
        $session_data['add_product_data']['price'] = '0.00';
        $session_data['add_product_data']['stock'] = null;
        
        // در هر سه حالت (فله، تعدادی، بسته) برای محصول متغیر باید واحد اصلی را بپرسیم
        // چون واحد در سطح محصول تعریف می‌شود نه ترکیب
        $session_data['state'] = 'awaiting_product_unit';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        $prompt = "";
        if ($mode === 'bulk') {
            $prompt = "لطفا واحد وزن فله را وارد کنید (مثلا: گرم، کیلوگرم، تن):";
        } elseif ($mode === 'package') {
            $prompt = "لطفا نوع بسته‌بندی را وارد کنید (مثلا: قوطی، جعبه):";
        } else {
            $prompt = "لطفا واحد شمارش را وارد کنید (مثلا: عدد، دستگاه):";
        }

        sendMessage(
            $chat_id, 
            $prompt,
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
            $platform
        );
    }
}

function handleProductPrice(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $price_str = trim($message['text'] ?? '');
    if (isValidNonNegativeNumber($price_str)) {
        $session_data['add_product_data']['price'] = convertPersianToEnglishDigits($price_str);
        $session_data['state'] = 'awaiting_product_stock';
        updateUserField($user_id, 'session', json_encode($session_data));
        sendMessage(
            $chat_id, 
            "قیمت ثبت شد. اکنون موجودی محصول را وارد کنید (عدد انگلیسی یا کلمه `null` برای نامحدود):".
            "\n(مثال: `10` یا `5.5` یا `null`)", 
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
            $platform
        );
    } else {
         sendMessage($chat_id, "لطفا قیمت را به صورت عدد صحیح یا اعشاری معتبر وارد کنید (مثال: `150000`).", null, $platform);
    }
}

function handleProductStock(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $stock_str = trim($message['text'] ?? '');
    $stock_value = null;
    $valid_input = false;

    if (strtolower($stock_str) === 'null') {
        $stock_value = null;
        $valid_input = true;
    } elseif (isValidNonNegativeNumber($stock_str)) {
        $stock_value = convertPersianToEnglishDigits($stock_str);
        $valid_input = true;
    }

    if ($valid_input) {
        $session_data['add_product_data']['stock'] = $stock_value;
        
        $mode = $session_data['add_product_data']['mode'] ?? 'count';
        
        // اینجا همیشه باید واحد را بپرسیم، چه فله باشد چه تعدادی چه بسته
        $session_data['state'] = 'awaiting_product_unit';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        $prompt = "";
        if ($mode === 'bulk') {
            $prompt = "موجودی ثبت شد. لطفا واحد وزن فله را وارد کنید (مثلا: گرم، کیلوگرم):";
        } elseif ($mode === 'package') {
            $prompt = "موجودی ثبت شد. لطفا نوع بسته‌بندی را وارد کنید (مثلا: بسته، قوطی):";
        } else {
            $prompt = "موجودی ثبت شد. لطفا واحد شمارش را وارد کنید (مثلا: عدد، دستگاه):";
        }

        sendMessage(
            $chat_id, 
            $prompt,
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true], 
            $platform
        );

    } else {
        sendMessage($chat_id, "لطفا موجودی را به صورت عدد معتبر (مثلا `10` یا `5.5` یا `0`) یا کلمه `null` وارد کنید.", null, $platform);
    }
}

function handleProductUnit(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $unit = trim($message['text'] ?? '');
    if (!empty($unit)) {
        $session_data['add_product_data']['unit'] = strip_tags($unit);
        
        $product_type = $session_data['add_product_data']['type'] ?? 'simple';
        $mode = $session_data['add_product_data']['mode'] ?? 'count';
        
        if ($product_type === 'variable') {
            // برای متغیر، بعد از واحد، میرویم سراغ ویژگی ها
            $session_data['state'] = 'awaiting_attribute_selection';
            updateUserField($user_id, 'session', json_encode($session_data));
            askForProductAttributes($chat_id, $user_id, $platform, $session_data, null);

        } else {
            // محصول ساده
            if ($mode === 'bulk') {
                // اگر فله است، واحد گرفته شده و وزن ثابت ندارد. ذخیره کن.
                handleProductWeightUnitAndSave(['text' => 'skip (auto)'], $chat_id, $user_id, $platform, $session_data);
            } elseif ($mode === 'count') {
                // اگر تعدادی است، وزن ندارد. ذخیره کن.
                handleProductWeightUnitAndSave(['text' => 'skip (auto)'], $chat_id, $user_id, $platform, $session_data);
            } else {
                // اگر بسته است (package)، وزن را بپرس
                $session_data['state'] = 'awaiting_product_weight';
                updateUserField($user_id, 'session', json_encode($session_data));
                
                sendMessage(
                    $chat_id, 
                    "واحد '{$unit}' ثبت شد. لطفا وزن خالص هر {$unit} را وارد کنید (فقط عدد، مثال: 4.6):",
                    ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
                    $platform
                );
            }
        }
    } else {
        sendMessage($chat_id, "واحد شمارش محصول نمی‌تواند خالی باشد.", null, $platform);
    }
}

function handleProductWeight(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $weight_str = trim($message['text'] ?? '');
    
    if (isValidPositiveNumber($weight_str)) {
         $session_data['add_product_data']['weight'] = convertPersianToEnglishDigits($weight_str);
        $session_data['state'] = 'awaiting_product_weight_unit';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        sendMessage(
            $chat_id,
            "وزن `{$weight_str}` ثبت شد. اکنون واحد وزن را وارد کنید (مثال: گرم، کیلوگرم):",
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
            $platform
        );
    } else {
        sendMessage($chat_id, "لطفا یک عدد معتبر برای وزن وارد کنید.", null, $platform);
    }
}

function handleProductWeightUnitAndSave(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $pdata = $session_data['add_product_data'] ?? [];
    
    if (isset($pdata['weight']) && $pdata['weight'] !== null) {
        $weight_unit = trim($message['text'] ?? '');
        if (empty($weight_unit)) {
            sendMessage($chat_id, "واحد وزن نمی‌تواند خالی باشد (مثال: گرم).", null, $platform);
            return;
        }
        $pdata['weight_unit'] = strip_tags($weight_unit);
    }

    if (empty($pdata['type']) || !isset($pdata['category_id']) || empty($pdata['image_path']) || 
        empty($pdata['name']) || !isset($pdata['unit'])) 
    {
         sendMessage($chat_id, "❌ خطا: اطلاعات محصول کامل نیست. لطفا از ابتدا شروع کنید.", ['remove_keyboard' => true], $platform);
         updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'add_product_data' => []]));
         showAdminMainMenu($chat_id, $user_id, $platform);
         return;
    }

    $product_id = createProduct(
        $pdata['name'], 
        $pdata['description'] ?? '', 
        $pdata['price'] ?? '0.00',
        $pdata['unit'], 
        $pdata['stock'] ?? null,
        $pdata['category_id'],
        $pdata['image_path'], 
        $pdata['type'],
        $pdata['weight'] ?? null,
        $pdata['weight_unit'] ?? null
    );

    if ($product_id > 0) {
        
        if ($pdata['type'] === 'variable') {
            $all_values_to_link = [];
            if (!empty($pdata['selected_values'])) {
                foreach ($pdata['selected_values'] as $attr_id => $values) {
                    $all_values_to_link = array_merge($all_values_to_link, $values);
                }
                setSelectedValuesForProduct($product_id, $all_values_to_link);
            }
            
            if (!empty($pdata['attributes'])) {
                foreach($pdata['attributes'] as $attr_id) {
                    addAttributeToProduct($product_id, $attr_id);
                }
            }
            
            $all_variations_data = $pdata['all_variations_data'] ?? [];
            foreach ($all_variations_data as $var_data) {
                $details = $var_data['details'];
                $options = $var_data['options'];
                
                $var_id = createProductVariation(
                    $product_id,
                    $details['price'] ?? null,
                    $details['stock'] ?? null,
                    $details['weight'] ?? null,
                    $details['weight_unit'] ?? null,
                    $details['image_path'] ?? null
                );
                
                if ($var_id > 0 && !empty($options)) {
                    foreach ($options as $value_id) {
                        addOptionToVariation($var_id, $value_id);
                    }
                }
            }
        }

          sendMessage($chat_id, "✅ محصول '{$pdata['name']}' با شناسه {$product_id} با موفقیت اضافه شد.", ['remove_keyboard' => true], $platform);
          
         updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'add_product_data' => []]));
          showAdminMainMenu($chat_id, $user_id, $platform);
    } else {
          sendMessage($chat_id, "❌ خطایی در افزودن محصول به دیتابیس رخ داد.", ['remove_keyboard' => true], $platform);
          updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'add_product_data' => []]));
          showAdminMainMenu($chat_id, $user_id, $platform);
     }
}


// --- توابع کمکی برای انتخاب ویژگی‌های محصول متغیر ---

function askForProductAttributes(int $chat_id, int $user_id, string $platform, array $session_data, ?int $message_id): void {
    $attributes = getAttributes();
    if (empty($attributes)) {
        sendMessage($chat_id, "خطا: هیچ ویژگی سراسری (مانند رنگ، سایز) تعریف نشده است. لطفا ابتدا از پنل مدیریت، بخش «مدیریت ویژگی‌ها» اقدام کنید.", ['remove_keyboard' => true], $platform);
        updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
        showAdminMainMenu($chat_id, $user_id, $platform);
        return;
    }
    
    $selected_attributes = $session_data['add_product_data']['attributes'] ?? [];
    
    $buttons = [];
    foreach($attributes as $attr) {
        $is_selected = in_array($attr['id'], $selected_attributes);
        $text = ($is_selected ? '✅ ' : '') . $attr['name'];
        $buttons[] = [['text' => $text, 'callback_data' => buildCallbackData('toggleAttr', ['attr_id' => $attr['id']])]];
    }
    
    if (!empty($selected_attributes)) {
        $buttons[] = [['text' => '➡️ ادامه (انتخاب مقادیر)', 'callback_data' => buildCallbackData('confirmAttributes')]];
    }
    
    $buttons[] = [['text' => '🔙 بازگشت (انتخاب نوع)', 'callback_data' => buildCallbackData('backToProductType')]];
    
    $text = "لطفا ویژگی‌هایی که این محصول متغیر دارد را انتخاب کنید (مثلا: رنگ و سایز):";
    $reply_markup = ['inline_keyboard' => $buttons];

    if ($message_id) {
        editMessage($chat_id, $message_id, $text, $reply_markup);
    } else {
        sendMessage($chat_id, $text, $reply_markup, $platform);
    }
}

function handleToggleAttribute(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $attribute_id = (int)($params['attr_id'] ?? 0);
    if ($attribute_id <= 0) return;

    $selected_attributes = $session_data['add_product_data']['attributes'] ?? [];
    
    if (in_array($attribute_id, $selected_attributes)) {
        $selected_attributes = array_diff($selected_attributes, [$attribute_id]);
    } else {
        $selected_attributes[] = $attribute_id;
    }
    
    $session_data['add_product_data']['attributes'] = array_values($selected_attributes);
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForProductAttributes($chat_id, $user_id, $platform, $session_data, $message_id);
}

function handleConfirmAttributes(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $selected_attributes = $session_data['add_product_data']['attributes'] ?? [];
    if (empty($selected_attributes)) {
        apiRequest('answerCallbackQuery', ['callback_query_id' => $GLOBALS['update']['callback_query']['id'] ?? null, 'text' => 'حداقل یک ویژگی باید انتخاب شود.', 'show_alert' => true], $platform);
        return;
    }
    
    $session_data['state'] = 'awaiting_value_selection';
    $session_data['add_product_data']['current_attr_index'] = 0;
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForAttributeValues($chat_id, $user_id, $platform, $session_data, $message_id);
}

function askForAttributeValues(int $chat_id, int $user_id, string $platform, array $session_data, ?int $message_id): void {
    $attributes = $session_data['add_product_data']['attributes'] ?? [];
    $index = $session_data['add_product_data']['current_attr_index'] ?? 0;
    
    if (!isset($attributes[$index])) {
        generateVariations($chat_id, $user_id, $platform, $session_data, $message_id);
        return;
    }
    
    $attribute_id = $attributes[$index];
    $stmt_attr = pdo()->prepare("SELECT name FROM attributes WHERE id = ?");
    $stmt_attr->execute([$attribute_id]);
    $attr_name = $stmt_attr->fetchColumn();
    
    $all_values = getAttributeValues($attribute_id);
    $selected_values = $session_data['add_product_data']['selected_values'][$attribute_id] ?? [];
    
    if (empty($all_values)) {
        editMessage($chat_id, $message_id, "خطا: ویژگی '{$attr_name}' هیچ مقداری (مانند قرمز، آبی) ندارد. لطفا ابتدا مقادیر را در بخش «مدیریت ویژگی‌ها» اضافه کنید.", ['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => buildCallbackData('backToAttrSelection')]]]]);
        return;
    }
    
    $buttons = [];
    $row = [];
    foreach($all_values as $value) {
        $is_selected = in_array($value['id'], $selected_values);
        $text = ($is_selected ? '✅ ' : '') . $value['value'];
        $row[] = ['text' => $text, 'callback_data' => buildCallbackData('toggleAttrValue', ['val_id' => $value['id']])];
        
        if (count($row) == 2) {
            $buttons[] = $row;
            $row = [];
        }
    }
    if (!empty($row)) $buttons[] = $row;
    
    if (!empty($selected_values)) {
        $buttons[] = [['text' => '➡️ ادامه', 'callback_data' => buildCallbackData('confirmAttrValues')]];
    }
    
    $buttons[] = [['text' => '🔙 بازگشت (انتخاب ویژگی‌ها)', 'callback_data' => buildCallbackData('backToAttrSelection')]];

    $text = "لطفا مقادیر مورد نظر برای ویژگی '{$attr_name}' را انتخاب کنید (مثلا فقط رنگ‌های موجود برای این محصول):";
    $reply_markup = ['inline_keyboard' => $buttons];
    
    if ($message_id) {
        editMessage($chat_id, $message_id, $text, $reply_markup);
    } else {
        sendMessage($chat_id, $text, $reply_markup, $platform);
    }
}

function handleToggleValue(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $value_id = (int)($params['val_id'] ?? 0);
    $index = $session_data['add_product_data']['current_attr_index'] ?? 0;
    $attribute_id = $session_data['add_product_data']['attributes'][$index] ?? 0;
    
    if ($value_id <= 0 || $attribute_id <= 0) return;

    $selected_values = $session_data['add_product_data']['selected_values'][$attribute_id] ?? [];
    
    if (in_array($value_id, $selected_values)) {
        $selected_values = array_diff($selected_values, [$value_id]);
    } else {
        $selected_values[] = $value_id;
    }
    
    $session_data['add_product_data']['selected_values'][$attribute_id] = array_values($selected_values);
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForAttributeValues($chat_id, $user_id, $platform, $session_data, $message_id);
}

function handleConfirmValues(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $index = $session_data['add_product_data']['current_attr_index'] ?? 0;
    $attribute_id = $session_data['add_product_data']['attributes'][$index] ?? 0;
    $selected_values = $session_data['add_product_data']['selected_values'][$attribute_id] ?? [];
    
    if (empty($selected_values)) {
        apiRequest('answerCallbackQuery', ['callback_query_id' => $GLOBALS['update']['callback_query']['id'] ?? null, 'text' => 'حداقل یک مقدار باید انتخاب شود.', 'show_alert' => true], $platform);
        return;
    }
    
    $session_data['add_product_data']['current_attr_index'] = $index + 1;
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForAttributeValues($chat_id, $user_id, $platform, $session_data, $message_id);
}

function handleBackToAttrSelection(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    $session_data['state'] = 'awaiting_attribute_selection';
    $session_data['add_product_data']['current_attr_index'] = 0;
    $session_data['add_product_data']['selected_values'] = [];
    updateUserField($user_id, 'session', json_encode($session_data));
    
    askForProductAttributes($chat_id, $user_id, $platform, $session_data, $message_id);
}

function handleBackToProductType(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform, array $session_data): void {
    if ($message_id) {
        apiRequest('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id], $platform);
    }
    startAddProductFlow($chat_id, $user_id, $platform);
}

function generateVariations(int $chat_id, int $user_id, string $platform, array $session_data, ?int $message_id): void {
    $selected_values_map = $session_data['add_product_data']['selected_values'] ?? [];
    if (empty($selected_values_map)) {
        editMessage($chat_id, $message_id, "خطا: هیچ مقداری انتخاب نشده است.", ['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => buildCallbackData('backToAttrSelection')]]]]);
        return;
    }
    
    $value_groups = array_values($selected_values_map);
    
    $combinations = [[]];
    foreach ($value_groups as $group) {
        $new_combinations = [];
        foreach ($combinations as $combination) {
            foreach ($group as $value_id) {
                $new_combinations[] = array_merge($combination, [$value_id]);
            }
        }
        $combinations = $new_combinations;
    }

    $session_data['add_product_data']['generated_combinations'] = $combinations;
    $session_data['add_product_data']['current_variation_index'] = 0;
    
    $session_data['state'] = 'awaiting_variation_details';
    updateUserField($user_id, 'session', json_encode($session_data));
    
    if ($message_id) {
        apiRequest('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id], $platform);
    }

    askForVariationDetails($chat_id, $user_id, $platform, $session_data);
}

function askForVariationDetails(int $chat_id, int $user_id, string $platform, array $session_data): void {
    $combinations = $session_data['add_product_data']['generated_combinations'] ?? [];
    $index = $session_data['add_product_data']['current_variation_index'] ?? 0;
    
    if (!isset($combinations[$index])) {
        // همه ترکیب‌ها تمام شده، حالا محصول اصلی ساخته می‌شود.
        $session_data['state'] = 'saving_variable_product';
        updateUserField($user_id, 'session', json_encode($session_data));
        // اینجا به جای skip، وزن و واحد را از session می‌خواند (چون قبلاً ست شده)
        handleProductWeightUnitAndSave(['text'=>'skip'], $chat_id, $user_id, $platform, $session_data);
        return;
    }
    
    $current_combination_ids = $combinations[$index];
    $value_names = [];
    foreach ($current_combination_ids as $value_id) {
        $stmt_val = pdo()->prepare("SELECT value FROM attribute_values WHERE id = ?");
        $stmt_val->execute([$value_id]);
        $value_names[] = $stmt_val->fetchColumn() ?: '?';
    }
    $combination_name = implode(' - ', $value_names);
    
    $total = count($combinations);
    $current_num = $index + 1;
    
    $session_data['state'] = 'awaiting_variation_price';
    updateUserField($user_id, 'session', json_encode($session_data));
    
    sendMessage(
        $chat_id,
        "لطفا جزئیات ترکیب {$current_num} از {$total} را وارد کنید:\n\n*{$combination_name}*\n\n" .
        "مرحله ۱: *قیمت* را به تومان وارد کنید (مثال: 150000)",
        ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
        $platform
    );
}

function handleVariationPrice(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $price_str = trim($message['text'] ?? '');
    if (isValidNonNegativeNumber($price_str)) {
        $session_data['add_product_data']['temp_variation']['price'] = convertPersianToEnglishDigits($price_str);
        $session_data['state'] = 'awaiting_variation_stock';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        sendMessage(
             $chat_id,
            "مرحله ۲: *موجودی* را وارد کنید (عدد، یا 'null' برای نامحدود)",
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
            $platform
        );
    } else {
        sendMessage($chat_id, "لطفا قیمت را به صورت عدد معتبر وارد کنید.", null, $platform);
    }
}

function handleVariationStock(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $stock_str = trim($message['text'] ?? '');
    $stock_value = null;
    $valid_input = false;

    if (strtolower($stock_str) === 'null') {
        $stock_value = null;
        $valid_input = true;
    } elseif (isValidNonNegativeNumber($stock_str)) {
        $stock_value = convertPersianToEnglishDigits($stock_str);
        $valid_input = true;
    }

    if ($valid_input) {
        $session_data['add_product_data']['temp_variation']['stock'] = $stock_value;
        
        $mode = $session_data['add_product_data']['mode'] ?? 'count';

        // هوشمند سازی بر اساس مد
        if ($mode === 'bulk') {
             // فله: وزن نال
             $session_data['add_product_data']['temp_variation']['weight'] = null;
             $session_data['add_product_data']['temp_variation']['weight_unit'] = null;
             // برو به عکس
             $session_data['state'] = 'awaiting_variation_photo';
             updateUserField($user_id, 'session', json_encode($session_data));
             sendMessage(
                $chat_id,
                "مرحله ۳: *عکس* مخصوص این ترکیب را ارسال کنید (اختیاری، برای رد شدن 'null' را بفرستید)",
                ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
                $platform
            );

        } elseif ($mode === 'count') {
             // تعدادی: وزن نال
             $session_data['add_product_data']['temp_variation']['weight'] = null;
             $session_data['add_product_data']['temp_variation']['weight_unit'] = null;
             // برو به عکس
             $session_data['state'] = 'awaiting_variation_photo';
             updateUserField($user_id, 'session', json_encode($session_data));
             sendMessage(
                $chat_id,
                "مرحله ۳: *عکس* مخصوص این ترکیب را ارسال کنید (اختیاری، برای رد شدن 'null' را بفرستید)",
                ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
                $platform
            );
        } else {
            // بسته: وزن بپرس
            $session_data['state'] = 'awaiting_variation_weight';
            updateUserField($user_id, 'session', json_encode($session_data));
            sendMessage(
                $chat_id,
                "مرحله ۳: *وزن* را وارد کنید (عدد، مثال: 5.2)",
                ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
                $platform
            );
        }
    } else {
        sendMessage($chat_id, "لطفا موجودی را به صورت عدد معتبر یا 'null' وارد کنید.", null, $platform);
    }
}

function handleVariationWeight(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $weight_str = trim($message['text'] ?? '');
    
    if (isValidPositiveNumber($weight_str)) {
        $session_data['add_product_data']['temp_variation']['weight'] = convertPersianToEnglishDigits($weight_str);
        $session_data['state'] = 'awaiting_variation_weight_unit';
        updateUserField($user_id, 'session', json_encode($session_data));
        
        sendMessage(
            $chat_id,
            "مرحله ۴: *واحد وزن* را وارد کنید (مثال: g یا kg)",
            ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
            $platform
        );
    } else {
        sendMessage($chat_id, "لطفا وزن را به صورت عدد معتبر وارد کنید.", null, $platform);
    }
}

function handleVariationWeightUnit(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $unit_str = trim($message['text'] ?? '');
    
    if (empty($unit_str) || strtolower($unit_str) === 'null') {
        sendMessage($chat_id, "واحد وزن نمی‌تواند خالی باشد (مثال: g).", null, $platform);
        return;
    }

    $session_data['add_product_data']['temp_variation']['weight_unit'] = strip_tags($unit_str);
    $session_data['state'] = 'awaiting_variation_photo';
    updateUserField($user_id, 'session', json_encode($session_data));
    
    sendMessage(
        $chat_id,
        "مرحله ۵: *عکس* مخصوص این ترکیب را ارسال کنید (اختیاری، برای رد شدن 'null' را بفرستید)",
        ['keyboard' => [[['text' => 'انصراف و بازگشت به منو']]], 'resize_keyboard' => true],
        $platform
    );
}

function handleVariationPhoto(array $message, int $chat_id, int $user_id, string $platform, array $session_data): void {
    $message_text = trim($message['text'] ?? '');
    $file_content = null;
    $saved_filename = null;

    if (isset($message['photo'])) {
        $photo_array = $message['photo'];
        $source_file_id = end($photo_array)['file_id'];
        sendMessage($chat_id, "⏳ در حال پردازش و ذخیره‌سازی تصویر...", null, $platform);
        $file_content = downloadFileContent($source_file_id, $platform);
    } elseif (strtolower($message_text) === 'null') {
        $saved_filename = null;
    }

    if ($file_content) {
        try {
            $storage_path = PRODUCT_IMAGE_STORAGE_PATH;
            if (!is_dir($storage_path) || !is_writable($storage_path)) {
                if (!@mkdir($storage_path, 0775, true)) {
                    throw new Exception("File storage path is not writable or cannot be created: {$storage_path}");
                }
            }
            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $mime_type = $finfo->buffer($file_content) ?: 'image/jpeg';
            $extension = 'jpg';
            if (strpos($mime_type, 'png') !== false) $extension = 'png';
            elseif (strpos($mime_type, 'gif') !== false) $extension = 'gif';
            elseif (strpos($mime_type, 'webp') !== false) $extension = 'webp';
            
            $new_filename = 'var_' . uniqid('', true) . '.' . $extension;
            $local_save_path = rtrim($storage_path, '/') . '/' . $new_filename;
            
            if (file_put_contents($local_save_path, $file_content) === false) {
                unset($file_content);
                throw new Exception("Failed to write temporary file: {$local_save_path}");
            }
            unset($file_content);
            $saved_filename = $new_filename;
        } catch (Exception $e) {
            if (defined('ERROR_LOG_PATH')) file_put_contents(ERROR_LOG_PATH, date('Y-m-d H:i:s') . " - handleVariationPhoto Save Error: " . $e->getMessage() . "\n", FILE_APPEND);
            sendMessage($chat_id, "❌ خطا در ذخیره‌سازی عکس. لطفا دوباره تلاش کنید.", null, $platform);
            return;
        }
    } elseif (!isset($message['photo']) && strtolower($message_text) !== 'null') {
        sendMessage($chat_id, "لطفا فقط عکس ارسال کنید یا 'null' را بفرستید.", null, $platform);
        return;
    }
    
    $session_data['add_product_data']['temp_variation']['image_path'] = $saved_filename;
    
    saveVariationAndContinue($chat_id, $user_id, $platform, $session_data);
}

function saveVariationAndContinue(int $chat_id, int $user_id, string $platform, array $session_data): void {
    $temp_variation = $session_data['add_product_data']['temp_variation'] ?? [];
    $combinations = $session_data['add_product_data']['generated_combinations'] ?? [];
    $index = $session_data['add_product_data']['current_variation_index'] ?? 0;
    
    $session_data['add_product_data']['all_variations_data'][] = [
        'details' => $temp_variation,
        'options' => $combinations[$index]
    ];
    
    $session_data['add_product_data']['current_variation_index'] = $index + 1;
    $session_data['add_product_data']['temp_variation'] = [];
    
    if (!isset($combinations[$index + 1])) {
        // پایان ترکیب‌ها. حالا محصول اصلی ساخته می‌شود.
        sendMessage($chat_id, "✅ تمام ترکیب‌ها تنظیم شدند.\n\nدرحال ذخیره محصول...", ['remove_keyboard' => true], $platform);
        handleProductWeightUnitAndSave(['text' => 'skip'], $chat_id, $user_id, $platform, $session_data);
    } else {
        updateUserField($user_id, 'session', json_encode($session_data));
        askForVariationDetails($chat_id, $user_id, $platform, $session_data);
    }
}

?>