<?php
if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}

function askForQuantity(array $params, int $chat_id, ?int $message_id, int $user_id, string $platform): void {
    $product_id = (int)($params['id'] ?? 0);
    $variation_id = (int)($params['var_id'] ?? 0);
    $variation_id = ($variation_id > 0) ? $variation_id : null;
    
    $cb_id = $GLOBALS['update']['callback_query']['id'] ?? null;

    if ($product_id <= 0) {
        if ($cb_id) apiRequest('answerCallbackQuery', ['callback_query_id' => $cb_id, 'text' => 'خطا: محصول نامشخص.', 'show_alert' => true], $platform);
        return;
    }

    $product = getProduct($product_id);
    if (!$product) {
        if ($cb_id) apiRequest('answerCallbackQuery', ['callback_query_id' => $cb_id, 'text' => 'خطا: محصول یافت نشد.', 'show_alert' => true], $platform);
        return;
    }

    $display_name = htmlspecialchars($product['name'] ?? 'نامشخص');
    $display_unit = $product['unit'] ?? 'عدد'; // واحد اصلی محصول (مثلاً بسته، گرم، کیلو)
    
    // متغیرهای کمکی برای تشخیص سناریوی وزن
    $current_weight = $product['weight'];
    $current_weight_unit = $product['weight_unit'];

    if ($product['type'] === 'variable' && $variation_id === null) {
        if ($cb_id) apiRequest('answerCallbackQuery', ['callback_query_id' => $cb_id, 'text' => 'لطفا ابتدا گزینه‌های محصول را انتخاب کنید.', 'show_alert' => true], $platform);
        return;
    }
    
    if ($variation_id !== null) {
        $variation = getVariationById($variation_id);
        if (!$variation) {
            if ($cb_id) apiRequest('answerCallbackQuery', ['callback_query_id' => $cb_id, 'text' => 'خطا: ترکیب محصول یافت نشد.', 'show_alert' => true], $platform);
            return;
        }
        $display_name .= ' (' . htmlspecialchars($variation['full_name']) . ')';
        // اگر ترکیب وزن مخصوص داشت، جایگزین کن
        if ($variation['weight'] !== null) {
            $current_weight = $variation['weight'];
            $current_weight_unit = $variation['weight_unit'];
        }
    }

    $session_data = ['state' => 'awaiting_quantity', 'product_id_to_add' => $product_id, 'variation_id_to_add' => $variation_id];
    updateUserField($user_id, 'session', json_encode($session_data));

    if ($message_id) {
        apiRequest('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id], $platform);
    }

    $cancel_keyboard = ['keyboard' => [[['text' => 'انصراف']]], 'resize_keyboard' => true, 'one_time_keyboard' => true];
    
    // --- شروع منطق هوشمند ساخت پیام ---
    
    $prompt_message = "";
    $known_weight_units = ['g', 'gr', 'gram', 'گرم', 'kg', 'kilo', 'kilogram', 'کیلوگرم'];
    
    // سناریو ۲: وزن فله‌ای (Implicit)
    // شرط: وزن خالی است اما واحد شمارش کالا یکی از واحدهای وزنی است (مثلاً "گرم" یا "کیلوگرم")
    if ($current_weight === null && in_array(strtolower($display_unit), $known_weight_units)) {
        $prompt_message = "⚖️ چند *{$display_unit}* از محصول *{$display_name}* می‌خواهید؟\n\n(لطفا فقط عدد وارد کنید، مثلا: `100` یا `0.5`)";
    }
    // سناریو ۱: وزن صریح (Explicit)
    // شرط: وزن و واحد وزن مشخص شده است
    elseif ($current_weight !== null && $current_weight_unit !== null) {
        $weight_display = number_format((float)$current_weight, (strpos((string)$current_weight, '.') !== false) ? 2 : 0);
        $prompt_message = "📦 چند *{$display_unit}* از محصول *{$display_name}* می‌خواهید؟\n(وزن هر {$display_unit}: {$weight_display} {$current_weight_unit})\n\n(لطفا عدد وارد کنید، مثلا: `2` یا `5`)";
    }
    // سناریو ۳: بدون وزن / تعدادی (No Weight)
    // شرط: هیچ وزنی ندارد و واحد هم وزنی نیست (مثلاً "عدد"، "بسته"، "لایسنس")
    else {
        $prompt_message = "🔢 چند *{$display_unit}* از محصول *{$display_name}* می‌خواهید؟\n\n(لطفا عدد وارد کنید، مثلا: `1` یا `10`)";
    }

    sendMessage(
        $chat_id,
        $prompt_message,
        $cancel_keyboard,
        $platform
    );
}

function handleQuantityInput(array $message, int $chat_id, int $user_id, string $platform, array $session_data, array $user): void {
    // شروع بلاک ایمنی برای جلوگیری از قفل شدن ربات
    try {
        $product_id = $session_data['product_id_to_add'] ?? null;
        $variation_id = $session_data['variation_id_to_add'] ?? null;
        
        // اگر کاربر دکمه انصراف یا دستورات ربات را فرستاد، اولویت با خروج است
        $text_check = isset($message['text']) ? trim($message['text']) : '';
        if ($text_check === 'انصراف' || strpos($text_check, '/') === 0) {
             sendMessage($chat_id, "عملیات لغو شد.", ['remove_keyboard' => true], $platform);
             updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
             showCustomerMainMenu($chat_id, $user_id, $platform, $user);
             return;
        }

        $message_text = isset($message['text']) ? trim(convertPersianToEnglishDigits($message['text'])) : null;

        if (!$product_id) {
            throw new Exception("Product ID missing in session.");
        }

        $product = getProduct($product_id);
        if (!$product) {
            sendMessage($chat_id, "خطا: محصول یافت نشد.", ['remove_keyboard' => true], $platform);
            updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
            showCustomerMainMenu($chat_id, $user_id, $platform, $user);
            return;
        }

        $product_name_safe = htmlspecialchars($product['name'] ?? 'نامشخص');
        $product_unit_safe = $product['unit'] ?? 'واحد';
        $product_stock_safe = $product['stock'];
        
        $current_weight = $product['weight'];

        if ($variation_id !== null) {
            $variation = getVariationById($variation_id);
            if ($variation) {
                $product_stock_safe = $variation['stock'];
                $product_name_safe .= ' (' . htmlspecialchars($variation['full_name']) . ')';
                if ($variation['weight'] !== null) {
                    $current_weight = $variation['weight'];
                }
            } else {
                 sendMessage($chat_id, "خطا: ترکیب محصول یافت نشد.", ['remove_keyboard' => true], $platform);
                 updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
                 showCustomerMainMenu($chat_id, $user_id, $platform, $user);
                 return;
            }
        }

        if ($message_text !== null && isValidPositiveNumber($message_text)) {
            $quantity = $message_text;
            
            $known_weight_units = ['g', 'gr', 'gram', 'گرم', 'kg', 'kilo', 'kilogram', 'کیلوگرم'];
            // استفاده از mb_strtolower برای پشتیبانی از حروف فارسی
            $unit_lower = function_exists('mb_strtolower') ? mb_strtolower($product_unit_safe) : strtolower($product_unit_safe);
            $is_bulk_scenario = ($current_weight === null && in_array($unit_lower, $known_weight_units));
            
            if ($is_bulk_scenario) {
                if (strpos($quantity, '.') !== false) {
                    $parts = explode('.', $quantity);
                    if (strlen($parts[1]) > 1) {
                        sendMessage($chat_id, "❌ حداکثر **۱ رقم اعشار** مجاز است (مثلاً ۱.۵).", null, $platform);
                        return; // اینجا return می‌کنیم تا کاربر بتواند دوباره عدد صحیح وارد کند
                    }
                }
            } else {
                if (strpos($quantity, '.') !== false) {
                    sendMessage($chat_id, "❌ برای واحد '{$product_unit_safe}' عدد اعشاری مجاز نیست. لطفا عدد صحیح وارد کنید.", null, $platform);
                    return;
                }
            }

            // چک کردن موجودی
            if ($product_stock_safe === null || bccomp((string)$product_stock_safe, $quantity, 2) >= 0) {
                
                // تلاش برای افزودن به سبد خرید
                if (addToCart($user_id, $product_id, $variation_id, $quantity)) {
                    // موفقیت: وضعیت را حتماً به idle برمی‌گردانیم
                    updateUserField($user_id, 'session', json_encode(['state' => 'idle', 'product_id_to_add' => null, 'variation_id_to_add' => null]));
                    
                    $mainMenuKeyboard = getCustomerMainMenuKeyboard($user);
                    sendMessage($chat_id, "✅ `{$quantity}` {$product_unit_safe} از محصول `{$product_name_safe}` به سبد خرید اضافه شد.", $mainMenuKeyboard, $platform);

                    $buttons = [
                        [['text' => '🛒 مشاهده سبد خرید و پرداخت', 'callback_data' => buildCallbackData('viewCart')]],
                        [['text' => '🛍 ادامه خرید', 'callback_data' => buildCallbackData('viewCat', ['id' => $product['category_id'], 'offset' => 0])]]
                    ];
                    sendMessage($chat_id, "می‌توانید سبد خرید خود را ببینید یا به خرید ادامه دهید:", ['inline_keyboard' => $buttons], $platform);
                    
                } else {
                    // خطای دیتابیس در افزودن
                    throw new Exception("Failed to add item to cart (DB Error).");
                }
            } else {
                $stock_formatted = number_format((float)$product_stock_safe, (strpos((string)$product_stock_safe, '.') !== false) ? 1 : 0, '.', '');
                sendMessage($chat_id, "❌ موجودی کافی نیست. حداکثر: `{$stock_formatted}` {$product_unit_safe}", null, $platform);
            }
        } else {
            sendMessage($chat_id, "لطفا یک عدد معتبر و مثبت وارد کنید.", null, $platform);
        }

    } catch (Throwable $e) {
        // ⚠️ نجا‌ت‌دهنده: اگر هر خطایی رخ داد، کاربر را آزاد کن
        if (defined('ERROR_LOG_PATH')) { 
            file_put_contents(ERROR_LOG_PATH, date('Y-m-d H:i:s') . " - Critical Error in handleQuantityInput: " . $e->getMessage() . "\n", FILE_APPEND); 
        }
        
        sendMessage($chat_id, "❌ متاسفانه خطایی رخ داد. به منوی اصلی بازگشتید.", ['remove_keyboard' => true], $platform);
        
        // ریست کردن وضعیت کاربر به حالت آزاد
        updateUserField($user_id, 'session', json_encode(['state' => 'idle']));
        showCustomerMainMenu($chat_id, $user_id, $platform, $user);
    }
}
?>