<?php

// Prevent direct script access
if (basename(__FILE__) == basename($_SERVER['SCRIPT_FILENAME'])) {
    die('No direct script access allowed');
}

// --- Input Validation Helper Functions ---

/**
 * Validates the format of a discount code (uppercase alphanumeric).
 */
function validateDiscountCodeFormat(string $code): bool {
    // Allows uppercase letters (A-Z) and digits (0-9) only.
    // Adjust the regex if other characters are allowed.
    return (bool)preg_match('/^[A-Z0-9]+$/', $code); 
}

/**
 * Validates the format of an Iranian National ID code (exactly 10 digits).
 * Does not perform checksum validation.
 */
function validateNationalIdFormat(string $nid): bool {
    // Remove any non-digits and check length
    $cleaned_nid = preg_replace('/\D/', '', convertPersianToEnglishDigits($nid)); // Assumes FormattingHelpers is loaded
    return strlen($cleaned_nid) === 10;
}

/**
 * Validates the format of an Iranian postal code (exactly 10 digits).
 */
function validatePostalCodeFormat(string $postal_code): bool {
    // Remove any non-digits and check length
    $cleaned_pc = preg_replace('/\D/', '', convertPersianToEnglishDigits($postal_code)); // Assumes FormattingHelpers is loaded
    return strlen($cleaned_pc) === 10;
}

/**
 * Validates the format of a bank card number (exactly 16 digits).
 */
function validateCardNumberFormat(string $card_number): bool {
    // Remove any non-digits and check length
    $cleaned = preg_replace('/\D/', '', convertPersianToEnglishDigits($card_number)); // Assumes FormattingHelpers is loaded
    return strlen($cleaned) === 16; 
}

/**
 * Validates if a string represents a valid positive number (integer or decimal > 0).
 */
function isValidPositiveNumber(string $number_str): bool {
    $cleaned = convertPersianToEnglishDigits(trim($number_str)); // Assumes FormattingHelpers is loaded
    // Check if numeric and greater than zero using bcmath for precision
    return is_numeric($cleaned) && (bccomp($cleaned, '0', 10) > 0); 
}

/**
 * Validates if a string represents a valid non-negative number (integer or decimal >= 0).
 */
function isValidNonNegativeNumber(string $number_str): bool {
    $cleaned = convertPersianToEnglishDigits(trim($number_str)); // Assumes FormattingHelpers is loaded
    // Check if numeric and greater than or equal to zero using bcmath for precision
    return is_numeric($cleaned) && (bccomp($cleaned, '0', 10) >= 0); 
}

?>
